# PlanMate AI - Smart Planning with AI

## Overview

PlanMate AI is a web-based personal and business planning application that helps users create, manage, and track their goals with AI-powered assistance. The platform combines traditional task management with intelligent AI suggestions powered by Google's Gemini API to provide smart recommendations, task breakdowns, and progress tracking. Users can create plans, add steps/subtasks, track completion progress, and receive AI-driven insights to optimize their planning workflow.

## User Preferences

Preferred communication style: Simple, everyday language.

## System Architecture

### Frontend Architecture

**Framework & Build System:**
- React with TypeScript for type-safe component development
- Vite as the build tool and development server
- Wouter for lightweight client-side routing
- TanStack Query (React Query) for server state management and caching

**UI Component System:**
- shadcn/ui component library with Radix UI primitives
- Tailwind CSS for styling with custom design system
- Dark mode as primary theme with light mode support
- Mobile-first responsive design with PWA-style interactions
- Custom color palette based on Linear/Notion hybrid approach

**State Management:**
- React Query for server state (plans, steps, notifications, user data)
- React Context for theme management
- Local component state for UI interactions

**Key Design Decisions:**
- **Problem:** Need professional, productivity-focused UI
- **Solution:** Implemented shadcn/ui with custom design system combining Linear's precision and Notion's flexibility
- **Rationale:** Pre-built accessible components with customization flexibility, consistent design language

### Backend Architecture

**Server Framework:**
- Express.js as the HTTP server
- TypeScript for type safety across the stack
- Session-based authentication using Replit Auth with OpenID Connect

**API Design:**
- RESTful API endpoints organized by resource type (plans, steps, notifications)
- Route handlers in `/server/routes.ts`
- Authentication middleware protecting all data endpoints
- Standardized error handling and logging

**Authentication & Authorization:**
- Replit Auth integration using OpenID/Passport strategy
- Session management with connect-pg-simple (PostgreSQL session store)
- User identification via OpenID claims (`sub` field)
- Protected routes requiring authentication middleware

**Key Design Decisions:**
- **Problem:** Need secure user authentication in Replit environment
- **Solution:** Implemented Replit Auth with OpenID Connect and PostgreSQL session storage
- **Rationale:** Leverages Replit's built-in authentication, secure session management, seamless user experience

### Data Layer

**Database:**
- PostgreSQL via Neon serverless database
- Drizzle ORM for type-safe database queries and schema management
- WebSocket support for serverless connections

**Schema Design:**
- `users` - User profiles with plan limits and admin flags
- `plans` - User plans with name, description, and progress tracking
- `planSteps` - Individual steps/tasks within plans with status tracking
- `notifications` - User notifications for reminders and AI suggestions
- `aiLogs` - Tracking of AI interactions and usage
- `savedAiResponses` - Saved AI assistance responses with subject labels
- `sessions` - Session storage for authentication

**Data Access Pattern:**
- Storage interface (`IStorage`) abstracting database operations
- Centralized storage module (`/server/storage.ts`) for all data access
- Automatic progress calculation based on step completion status

**Key Design Decisions:**
- **Problem:** Need type-safe database operations with good developer experience
- **Solution:** Drizzle ORM with shared schema types between frontend and backend
- **Rationale:** Full TypeScript support, automatic type inference, migration support, better than raw SQL

### AI Integration

**AI Provider:**
- Google Gemini API (via `@google/genai` package)
- Configurable API key via environment variables
- AI suggestions for plan optimization and task breakdown

**AI Features:**
- Plan text rewriting for clarity
- Subtask generation and suggestions
- Progress summarization
- AI response logging for usage tracking
- Step-specific AI assistance with context-aware guidance
- Save and organize AI responses with custom labels
- AI response history with full-text search capabilities

**Key Design Decisions:**
- **Problem:** Need intelligent planning assistance
- **Solution:** Integrated Google Gemini API for contextual suggestions
- **Rationale:** Powerful language model, good for task planning domain, cost-effective API

## External Dependencies

### Third-Party Services

**Authentication:**
- Replit Auth (OpenID Connect) - User authentication and identity management
- Session storage in PostgreSQL

**Database:**
- Neon Serverless PostgreSQL - Primary data storage
- Connection via `@neondatabase/serverless` with WebSocket support

**AI Services:**
- Google Gemini API - AI-powered suggestions and content generation
- API key configuration via `GEMINI_API_KEY` environment variable

### Key Libraries

**Frontend:**
- `@tanstack/react-query` - Server state management and caching
- `@radix-ui/*` - Accessible UI primitives (dialogs, dropdowns, etc.)
- `wouter` - Lightweight routing
- `tailwindcss` - Utility-first CSS framework
- `date-fns` - Date formatting and manipulation

**Backend:**
- `express` - Web server framework
- `drizzle-orm` - Type-safe ORM
- `passport` - Authentication middleware
- `connect-pg-simple` - PostgreSQL session store
- `@google/genai` - Google Gemini AI SDK

**Development:**
- `vite` - Build tool and dev server
- `typescript` - Type checking
- `drizzle-kit` - Database migrations

### Environment Configuration

Required environment variables:
- `DATABASE_URL` - PostgreSQL connection string (Neon)
- `SESSION_SECRET` - Session encryption key
- `GEMINI_API_KEY` - Google AI API key
- `REPL_ID` - Replit deployment identifier
- `ISSUER_URL` - OpenID Connect issuer URL