import { Link } from "wouter";
import { Button } from "@/components/ui/button";
import { Card } from "@/components/ui/card";
import { 
  Sparkles, 
  Target, 
  Brain, 
  Zap, 
  Calendar,
  TrendingUp,
  CheckCircle,
  ArrowRight,
  Users,
  Star,
  Rocket
} from "lucide-react";
import { motion } from "framer-motion";

const fadeInUp = {
  initial: { opacity: 0, y: 60 },
  animate: { opacity: 1, y: 0 },
  transition: { duration: 0.6 }
};

const staggerContainer = {
  animate: {
    transition: {
      staggerChildren: 0.1
    }
  }
};

export default function Landing() {
  return (
    <div className="min-h-screen bg-background pt-16">

      {/* Hero Section */}
      <section className="relative overflow-hidden pt-32 pb-20 md:pt-40 md:pb-32">
        {/* Animated Background Gradient */}
        <div className="absolute inset-0 bg-gradient-to-br from-indigo-500/20 via-purple-500/20 to-teal-400/20 animate-gradient-xy"></div>
        
        {/* Floating Orbs */}
        <motion.div 
          className="absolute top-20 left-10 w-72 h-72 bg-indigo-500/30 rounded-full blur-3xl"
          animate={{
            scale: [1, 1.2, 1],
            opacity: [0.3, 0.5, 0.3],
          }}
          transition={{
            duration: 8,
            repeat: Infinity,
            ease: "easeInOut"
          }}
        />
        <motion.div 
          className="absolute bottom-20 right-10 w-96 h-96 bg-purple-500/30 rounded-full blur-3xl"
          animate={{
            scale: [1.2, 1, 1.2],
            opacity: [0.3, 0.5, 0.3],
          }}
          transition={{
            duration: 10,
            repeat: Infinity,
            ease: "easeInOut"
          }}
        />

        <div className="container mx-auto px-4 md:px-6 relative">
          <motion.div 
            className="max-w-5xl mx-auto text-center space-y-8"
            initial="initial"
            animate="animate"
            variants={staggerContainer}
          >
            <motion.div 
              className="inline-block"
              variants={fadeInUp}
            >
              <div className="inline-flex items-center gap-2 px-4 py-2 rounded-full bg-gradient-to-r from-indigo-500/10 to-purple-500/10 border border-indigo-500/20">
                <Sparkles className="h-4 w-4 text-indigo-400" />
                <span className="text-sm font-medium text-foreground/80">AI-Powered Planning Platform</span>
              </div>
            </motion.div>

            <motion.h1 
              className="text-5xl md:text-7xl lg:text-8xl font-bold leading-tight"
              variants={fadeInUp}
            >
              Transform Ideas Into{" "}
              <span className="bg-gradient-to-r from-indigo-400 via-purple-400 to-teal-400 bg-clip-text text-transparent">
                Reality
              </span>
            </motion.h1>

            <motion.p 
              className="text-xl md:text-2xl text-muted-foreground max-w-3xl mx-auto"
              variants={fadeInUp}
            >
              The smartest way to plan, track, and achieve your goals with AI assistance.
              Built for individuals and teams who dream big.
            </motion.p>

            <motion.div 
              className="flex flex-col sm:flex-row gap-4 justify-center items-center pt-4"
              variants={fadeInUp}
            >
              <Link href="/login">
                <Button 
                  size="lg" 
                  className="bg-gradient-to-r from-indigo-500 to-purple-600 px-8 py-6 text-lg shadow-2xl shadow-purple-500/30 hover:shadow-purple-500/50 transition-shadow"
                  data-testid="button-get-started"
                >
                  Start Free Today
                  <ArrowRight className="ml-2 h-5 w-5" />
                </Button>
              </Link>
              <Button 
                size="lg" 
                variant="outline" 
                className="px-8 py-6 text-lg backdrop-blur-sm border-2"
                onClick={() => document.getElementById('features')?.scrollIntoView({ behavior: 'smooth' })}
                data-testid="button-explore-demo"
              >
                <Rocket className="mr-2 h-5 w-5" />
                See How It Works
              </Button>
            </motion.div>

            {/* Stats */}
            <motion.div 
              className="grid grid-cols-3 gap-8 pt-16 max-w-2xl mx-auto"
              variants={fadeInUp}
            >
              <div className="text-center">
                <div className="text-3xl md:text-4xl font-bold bg-gradient-to-r from-indigo-400 to-purple-400 bg-clip-text text-transparent">10K+</div>
                <div className="text-sm text-muted-foreground mt-1">Active Users</div>
              </div>
              <div className="text-center">
                <div className="text-3xl md:text-4xl font-bold bg-gradient-to-r from-purple-400 to-pink-400 bg-clip-text text-transparent">50K+</div>
                <div className="text-sm text-muted-foreground mt-1">Plans Created</div>
              </div>
              <div className="text-center">
                <div className="text-3xl md:text-4xl font-bold bg-gradient-to-r from-teal-400 to-cyan-400 bg-clip-text text-transparent">95%</div>
                <div className="text-sm text-muted-foreground mt-1">Success Rate</div>
              </div>
            </motion.div>
          </motion.div>
        </div>
      </section>

      {/* Features Section */}
      <section id="features" className="py-24 bg-card/30 relative">
        <div className="container mx-auto px-4 md:px-6">
          <motion.div 
            className="text-center mb-16"
            initial={{ opacity: 0, y: 20 }}
            whileInView={{ opacity: 1, y: 0 }}
            viewport={{ once: true }}
            transition={{ duration: 0.6 }}
          >
            <h2 className="text-3xl md:text-5xl font-bold mb-4">
              Everything You Need to{" "}
              <span className="bg-gradient-to-r from-indigo-400 to-purple-400 bg-clip-text text-transparent">
                Succeed
              </span>
            </h2>
            <p className="text-muted-foreground text-lg max-w-2xl mx-auto">
              Powerful features designed to supercharge your productivity
            </p>
          </motion.div>

          <motion.div 
            className="grid grid-cols-1 md:grid-cols-2 lg:grid-cols-3 gap-8"
            variants={staggerContainer}
            initial="initial"
            whileInView="animate"
            viewport={{ once: true }}
          >
            {[
              {
                icon: Brain,
                title: "AI-Powered Suggestions",
                description: "Get intelligent recommendations and auto-complete your plans with Gemini AI",
                gradient: "from-indigo-500/20 to-purple-500/20",
                iconColor: "text-indigo-400",
                borderColor: "hover:border-purple-500/30"
              },
              {
                icon: Target,
                title: "Smart Goal Planning",
                description: "Break down complex goals into actionable steps with AI assistance",
                gradient: "from-cyan-500/20 to-teal-500/20",
                iconColor: "text-cyan-400",
                borderColor: "hover:border-cyan-500/30"
              },
              {
                icon: TrendingUp,
                title: "Progress Analytics",
                description: "Track your progress with beautiful visualizations and insights",
                gradient: "from-purple-500/20 to-pink-500/20",
                iconColor: "text-purple-400",
                borderColor: "hover:border-purple-500/30"
              },
              {
                icon: Zap,
                title: "Lightning Fast",
                description: "Built for speed with real-time updates and instant sync",
                gradient: "from-yellow-500/20 to-orange-500/20",
                iconColor: "text-yellow-400",
                borderColor: "hover:border-yellow-500/30"
              },
              {
                icon: Calendar,
                title: "Smart Scheduling",
                description: "AI helps you prioritize and schedule tasks efficiently",
                gradient: "from-teal-500/20 to-cyan-500/20",
                iconColor: "text-teal-400",
                borderColor: "hover:border-teal-500/30"
              },
              {
                icon: Users,
                title: "Team Collaboration",
                description: "Work together seamlessly with shared plans and updates",
                gradient: "from-indigo-500/20 to-blue-500/20",
                iconColor: "text-indigo-400",
                borderColor: "hover:border-indigo-500/30"
              }
            ].map((feature, index) => (
              <motion.div
                key={index}
                variants={fadeInUp}
              >
                <Card className={`p-8 bg-card/50 backdrop-blur-sm border-white/10 ${feature.borderColor} transition-all hover:shadow-2xl hover:-translate-y-1 group`}>
                  <motion.div 
                    className={`h-16 w-16 rounded-2xl bg-gradient-to-br ${feature.gradient} flex items-center justify-center mb-6`}
                    whileHover={{ scale: 1.1, rotate: 5 }}
                    transition={{ type: "spring", stiffness: 400, damping: 10 }}
                  >
                    <feature.icon className={`h-8 w-8 ${feature.iconColor}`} />
                  </motion.div>
                  <h3 className="text-xl font-semibold mb-3 group-hover:text-foreground transition-colors">{feature.title}</h3>
                  <p className="text-muted-foreground">
                    {feature.description}
                  </p>
                </Card>
              </motion.div>
            ))}
          </motion.div>
        </div>
      </section>

      {/* Social Proof Section */}
      <section className="py-24 relative overflow-hidden">
        <div className="container mx-auto px-4 md:px-6">
          <motion.div 
            className="text-center mb-16"
            initial={{ opacity: 0, y: 20 }}
            whileInView={{ opacity: 1, y: 0 }}
            viewport={{ once: true }}
            transition={{ duration: 0.6 }}
          >
            <h2 className="text-3xl md:text-5xl font-bold mb-4">
              Loved by{" "}
              <span className="bg-gradient-to-r from-indigo-400 to-purple-400 bg-clip-text text-transparent">
                Thousands
              </span>
            </h2>
            <p className="text-muted-foreground text-lg">What our users are saying</p>
          </motion.div>

          <motion.div 
            className="grid grid-cols-1 md:grid-cols-3 gap-8 max-w-6xl mx-auto"
            variants={staggerContainer}
            initial="initial"
            whileInView="animate"
            viewport={{ once: true }}
          >
            {[
              {
                quote: "PlanMate AI transformed how I manage my startup. The AI suggestions are spot-on!",
                author: "Sarah Chen",
                role: "Startup Founder",
                rating: 5
              },
              {
                quote: "Finally, a planning tool that actually understands what I need. Game changer!",
                author: "Michael Rodriguez",
                role: "Product Manager",
                rating: 5
              },
              {
                quote: "The progress tracking keeps me motivated. I've completed more goals this year than ever!",
                author: "Emily Watson",
                role: "Freelance Designer",
                rating: 5
              }
            ].map((testimonial, index) => (
              <motion.div
                key={index}
                variants={fadeInUp}
              >
                <Card className="p-6 bg-card/50 backdrop-blur-sm border-white/10 hover:border-purple-500/30 transition-all">
                  <div className="flex gap-1 mb-4">
                    {[...Array(testimonial.rating)].map((_, i) => (
                      <Star key={i} className="h-4 w-4 fill-yellow-400 text-yellow-400" />
                    ))}
                  </div>
                  <p className="text-foreground/90 mb-4 italic">"{testimonial.quote}"</p>
                  <div className="flex items-center gap-3">
                    <div className="h-10 w-10 rounded-full bg-gradient-to-br from-indigo-500 to-purple-600 flex items-center justify-center text-white font-semibold">
                      {testimonial.author[0]}
                    </div>
                    <div>
                      <div className="font-semibold text-sm">{testimonial.author}</div>
                      <div className="text-xs text-muted-foreground">{testimonial.role}</div>
                    </div>
                  </div>
                </Card>
              </motion.div>
            ))}
          </motion.div>
        </div>
      </section>

      {/* CTA Section */}
      <section className="py-24 relative overflow-hidden">
        <div className="absolute inset-0 bg-gradient-to-br from-indigo-500/10 via-purple-500/10 to-teal-400/10"></div>
        <motion.div 
          className="container mx-auto px-4 md:px-6 text-center relative"
          initial={{ opacity: 0, y: 20 }}
          whileInView={{ opacity: 1, y: 0 }}
          viewport={{ once: true }}
          transition={{ duration: 0.6 }}
        >
          <div className="max-w-3xl mx-auto space-y-8">
            <h2 className="text-4xl md:text-6xl font-bold leading-tight">
              Ready to{" "}
              <span className="bg-gradient-to-r from-indigo-400 via-purple-400 to-teal-400 bg-clip-text text-transparent">
                Achieve More?
              </span>
            </h2>
            <p className="text-xl text-muted-foreground max-w-2xl mx-auto">
              Join thousands of successful planners. Start your journey today, completely free.
            </p>
            <div className="flex flex-col sm:flex-row gap-4 justify-center items-center">
              <Link href="/login">
                <Button 
                  size="lg" 
                  className="bg-gradient-to-r from-indigo-500 to-purple-600 px-12 py-6 text-lg shadow-2xl shadow-purple-500/30"
                  data-testid="button-cta-signup"
                >
                  Get Started Free
                  <CheckCircle className="ml-2 h-5 w-5" />
                </Button>
              </Link>
            </div>
            <p className="text-sm text-muted-foreground">
              No credit card required • Free forever plan available
            </p>
          </div>
        </motion.div>
      </section>
    </div>
  );
}
